C      *****************************************************************
C      * Subroutine YASEDriver                                         *
C      * Provides the interface between the TreGro model and the YASEQL*
C      * soil model.  All variables passed to this routine from TreGro *
C      * are assigned to YASEQL variables within this routine so that  *
C      * it is clear which variables are being input to this routine   *
C      * and which are being output.                                   *
C      *                                                               *
C      * Variables:                                                    *
C      * BlkDens   -Global- A vector, of size NLayers, of the bulk     *
C      *                    density of the soil for each layer.  Input *
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g/cm**3.       *
C      *                    (Common block VModel, file VModel.f)       *
C      * CatchArea -Global- A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Units are m**2.                            *
C      *                    (Common block VModel, file VModel.f)       *
C      * ElapT     -Global- The elapsed time since the last call, i.e.,*
C      *                    the model timestep.  (This is model time,  *
C      *                    not real time.)  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are s.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * FName     -Global- The name of the file to be read in by      *
C      *                    the routine InDat. Input from TREGRO, or   *
C      *                    read into the model by the routine InParam.*
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * Frstim    -Global- A flag that is a 1 the first time the soil *
C      *                    model is called and is a 0 otherwise.      *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VModel, file VModel.f)       *
C      * I         -Local - Index variable.  Unitless.                 *
C      * J         -Local - Index variable.  Unitless.                 *
C      * KT        -Global- A matrix, of size NComp by NLayer, of the  *
C      *                    known total amount for each component, for *
C      *                    each layer.  The total includes the amount *
C      *                    of the component as a free component and as*
C      *                    part of all dependent species.  Units are  *
C      *                    mol.                                       *
C      *                    (Common block VModel, file VModel.f)       *
C      * LfNut     -Global- A vector, of size NComp, of the amount of  *
C      *                    each nutrient in the leaves that senesced  *
C      *                    during the day.  Units are g nutrients.    *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * LfSenesc  -Global- The amount of leaves that senesced during  *
C      *                    the day.  Units are g C.                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * Lyr       -Local - Index variable.  Unitless.                 *
C      * NLayers   -Global- The number of soil layers in the system.   *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Unitless.         *
C      *                    (Common block VPos, file VModel.f)         *
C      * NuTake    -Global- A matrix of size NComp by NLayers,  of the *
C      *                    nutrients taken up by the tree in TREGRO   *
C      *                    for each layer.  The order of nutrients is *
C      *                    found by using the array SpIndex.          *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Units are mol.    *
C      *                    (Common block VModel, file VModel.f)       *
C      * NUTTFConc -Global- A vector, of size  NComp, of throughfall   *
C      *                    concentrations for all nutrients (species  *
C      *                    of interest).  Units are mol/L.            *
C      *                    (Common block VModel, file VModel.f)       *
C      * Outfile   -Global- Name of output data file.  Input from      *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VModel, file VModel.f)       *              
C      * RtNut     -Global- A matrix, NLayers by NComps, of the        *
C      *                    amount of nutrients in the roots, in each  *
C      *                    layer, that senesced during the day.  Input*
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Units are g nutr.        *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * RtSenesc  -Global- A vector, of size NLayers, of the amount of*
C      *                    roots, in each layer, that senesced during *
C      *                    the day.  Input from TREGRO, or read into  *
C      *                    the model by the routine InParam.  Units   *
C      *                    are g C.                                   *
C      *                    (Common block VDecomp, file VModel.f)      *
C      * SLMoist   -Global- A vector, of size NLayers, of soil moisture*
C      *                    for each layer.  Input from TREGRO, or read*
C      *                    into the model by the routine InParam.     *
C      *                    Units are mm water.                        *
C      * SlTemp    -Global- A vector, of size NLayers, of soil         *
C      *                    temperatures for each layer.  Input from   *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Units are deg C.         *                     
C      * SoChem    -Output- An array, of size SpNum by NLayers (note   *
C      *                    that declared size must match SChem in     *
C      *                    TREGRO), of concentrations for all         *
C      *                    nutrients (species of interest), in all    *
C      *                    soil layers.  Units are mol/L.             *
C      * SoTot     -Output- An array, of size SpNum by NLayers, of the *
C      *                    total amount of each nutrient (species of  *
C      *                    interest) in each layer.  Units are mol.   *                                    *
C      * SpConc    -Global- A matrix, of size NSpecies by NLayers, of  *
C      *                    species concentrations, one set for each   *
C      *                    layer. Units are mol/L.                    *
C      *                    (Common block VModel, file VModel.f)       *
C      * SpIndex   -Global- A vector, of size SpNum, of indices into   *
C      *                    arrays which match each species of         *
C      *                    interest.  SpIndex(i) is an index that when*
C      *                    used in the Labels array will give the     *
C      *                    name of the species amount the tree has    *
C      *                    consumed is NuTake(i).  SpIndex(i) when    *
C      *                    used as the index into SpConc will give the*
C      *                    concentration of this species.  Input from *
C      *                    TREGRO, or read into the model by the      *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      * SpName    -Global- A vector, of size SpNum, of names of       *
C      *                    species of interest.  Input from TREGRO, or*
C      *                    read into the model by the routine InParam.*
C      *                    Unitless.                                  *
C      *                    (Common block VIF, file VModel.f)          *
C      * SpNum     -Global- The number of species of interest.  Input  *
C      *                    from TREGRO, or read into the model by the *
C      *                    routine InParam.  Unitless.                *
C      *                    (Common block VIF, file VModel.f)          *
C      * StndAlone -Global- Indicates whether soil model is running as *
C      *                    a stand alone model (TRUE) or not (FALSE). * 
C      *                    Unitless.                                  *
C      *                    (Common block VModel, file VModel.f)       *
C      * TFComp    -Global- The number of components in the            *
C      *                    throughfall.  This should be the same as   *
C      *                    the number of components.  Unitless.       *
C      *                    (Common block VModel, file VModel.f)       *
C      * TGBulk    -Input - A vector, of size NLayers, of the bulk     *
C      *                    density of the soil for each layer.  Input *
C      *                    from TREGRO.  Units are g/cm**3.           *
C      * TGCatch   -Input - A vector, of size NLayers, of the catchment*
C      *                    area for each layer.  Input from TREGRO.   *
C      *                    Units are m**2.                            *
C      * TGElap    -Input - The elapsed time since the last call, i.e.,*
C      *                    the model timesetp.  This is model time,   *
C      *                    not real time.  Input from TREGRO.  Units  *
C      *                    are s.                                     *
C      * TGFrstim  -Input - A flag that is a 1 the first time the soil *
C      *                    model is called and is a 0 otherwise.      *
C      *                    Input from TREGRO.  Unitless.              *
C      * TGInFil   -Input - The name of the file to be read in by      *
C      *                    the routine InDat. Input from TREGRO.      *
C      *                    Unitless.                                  *
C      * TGLay     -Input - The number of soil layers in the TREGRO    *
C      *                    model.  Unitless.                          *
C      * TGLfNut   -Input - A vector, of size NComp, of the amount of  *
C      *                    each nutrient in the leaves.  Units are    *
C      *                    g nutrients.                               *
C      * TGLfSen   -Input - The amount of leaves that senesced during  *
C      *                    the day.  Units are g C.                   *
C      * TGNumOut  -Input - The number of the output file.  Input from *
C      *                    TreGro.  Unitless.                         *
C      * TGNuts    -Input - An array, of size NComp by NLayers, of the *
C      *                    nutrients (species of interest) taken up by*
C      *                    the tree in TREGRO for each layer.  Input  *
C      *                    from TREGRO.  Units are mol.               *
C      * TGOutFil  -Input - Name of output data file.  Input from      *
C      *                    TREGRO.  Unitless.                         *
C      * TGRtNut   -Input - A matrix, NLayers by NComps, of the        *
C      *                    amount of nutrients in the roots in each   *
C      *                    layer.  Input from TREGRO.  Units are      *
C      *                    g nutrients.                               *
C      * TGRtSen   -Input - A vector, of size NLayers, of the amount of*
C      *                    roots per layer that senesced during the   *
C      *                    day.  Input from TREGRO.  Units are g C.   *
C      * TGSLMoist -Input - A vector, of size NLayers, of soil moisture*
C      *                    for each layer.  Input from TREGRO.  Units *
C      *                    are mm water.                              *
C      * TGSlTemp  -Input - A vector, of size NLayers, of soil         *
C      *                    temperatures for each layer.  Input from   *
C      *                    TREGRO.  Units are deg C.                  *
C      * TGSpInx   -Input - A vector, of size SpNum, of indices into   *
C      *                    arrays that match each species of          *
C      *                    interest.  SpIndex(i) is an index that when*
C      *                    used in the Labels array will give the     *
C      *                    name of the species amount the tree has    *
C      *                    consumed is NuTake(i).  SpIndex(i) when    *
C      *                    used as the index into SpConc will give the*
C      *                    concentration of this species.  Unitless.  *
C      * TGSpName  -Input - A vector, of size SpNum, of species of     *
C      *                    interest names.  Input from TREGRO.        *
C      *                    Unitless.                                  *
C      * TGSpNum   -Input - The number of species of interest.  Input  *
C      *                    from TREGRO.  Unitless.                    *
C      * TGTFConc  -Input - A vector, of size NComp, of throughfall    *
C      *                    concentrations.  The order of the          *
C      *                    components is the same as in SPConc.  Input*
C      *                    from TREGRO.  Units are mol/L.             *
C      * TGTheta   -Input - A vector, of size NLayers, of volumetric   *
C      *                    soil moisture  contents for each layer.    *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Units are         *
C      *                    cm**3/cm**3.                               *
C      * TGThick   -Input - A vector, of size NLayers, of thicknesses  *
C      *                    of each soil layer.  Input from TREGRO, or *
C      *                    read into the model by the routine InParam.*
C      *                    Units are m.                               *
C      * TGWFlux   -Input - A vector, of size NLayers, of net water    *
C      *                    fluxes between layers Flux(0) is the net   *
C      *                    flux between the atmosphere and the top    *
C      *                    layer, Flux(1) is the net flux between     *
C      *                    layers number 1 and 2, Flux(3) is the net  *
C      *                    flux between layers number 3 and 4.  If    *
C      *                    the flux is positive, the water is moving  *
C      *                    down; if the flux is negative, the water is*
C      *                    moving up.  Input from TREGRO, or read into*
C      *                    the model by the routine InParam.  Units   *
C      *                    are L/(m**2 sec).                          *                     
C      * Theta     -Global- A vector, of size NLayers, of volumetric   *
C      *                    soil moisture contents for each layer.     *
C      *                    Input from TREGRO, or read into the model  *
C      *                    by the routine InParam.  Units are         *
C      *                    cm**3/cm**3.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * Thickness -Global- A vector, of size NLayers, of thicknesses  *
C      *                    of each soil layer.  Input from TREGRO, or *
C      *                    read into the model by the routine InParam.*
C      *                    Units are m.                               *
C      *                    (Common block VModel, file VModel.f)       *
C      * UnitNum   -Local - The unit number for the soil model's output*
C      *                    file.  Unitless.                           *
C      * WFlux     -Global- A vector, of size NLayers, of net water    *
C      *                    fluxes between layers.  Flux(0) is the net *
C      *                    flux between the atmosphere and the top    *
C      *                    layer, Flux(1) is the net flux between     *
C      *                    layers number 1 and 2, Flux(3) is the net  *
C      *                    flux between layers number 3 and 4.  If    *
C      *                    the flux is positive, the water is moving  *
C      *                    down; if the flux is negative, the water is*
C      *                    moving up.  Input from TREGRO, or read into*
C      *                    the model by the routine InParam.  Units   *
C      *                    are L/(m**2 sec).                          *                     
C      *                    (Common block VModel, file VModel.f)       *
C      * YASOut    -Global- The output file number for the YASEQL      *
C      *                    model.  Unitless.                          *
C      *                    (Common block VModel, file VModel.f)       *
C      *****************************************************************
       SUBROUTINE YASEDRIVER(TGFRSTIM, TGINFIL, TGOUTFIL, TGNUMOUT,
     >                       TGELAP,   TGLAY,   TGBULK,   TGCATCH,  
     >                       TGTHETA,  TGTHICK, TGWFLUX,  TGTFCONC, 
     >                       TGSPNUM,  TGSPNAME,TGSPINX,  TGNUTS,
     >                       TGRTSEN,  TGLFSEN, TGSLTEMP, TGLFNUT,
     >                       TGRTNUT,  TGSLMOIST,
     >                       SOCHEM,   SOTOT)
							IMPLICIT NONE
							INCLUDE 'VMODEL.INC'
							INTEGER  GOAHEAD, I, J, LYR, TGFRSTIM, TGLAY 
       INTEGER  TGNUMOUT, TGSPINX(10), TGSPNUM
       REAL*8 SOCHEM(10,3),  SOTOT(10,3),    TGBULK(15)
       REAL*8 TGCATCH(15),   TGELAP,         TGLFNUT(20) 
       REAL*8 TGLFSEN,       TGNUTS(20,15),  TGRTNUT(20,15) 
       REAL*8 TGRTSEN(15),   TGSLMOIST(15),  TGSLTEMP(15) 
       REAL*8 TGTFCONC(20),  TGTHETA(15),    TGTHICK(15)
       REAL*8 TGWFLUX(0:15)
       CHARACTER*8  TGSPNAME(10)
       CHARACTER*255 TGINFIL, TGOUTFIL

C      *------------------------------------------------------------*
C      * Indicate that soil model is being called by another model. *
C      *------------------------------------------------------------*
       STNDALONE = .FALSE.

C      *--------------------------*
C      * Set the first time flag. *
C      *--------------------------*
       FRSTIM = TGFRSTIM

C      *------------------------------------------------------------*
C      * Assign the input and output data file names and the output *
C      * file number.                                               *
C      *------------------------------------------------------------*
       FNAME   = TGINFIL
       OUTFILE = TGOUTFIL
       YASOUT  = TGNUMOUT

C      *-------------------------------------------*
C      * Set the elapsed time since the last call. *
C      *-------------------------------------------*
       ELAPT = TGELAP

C      *---------------------------*
C      * Set the number of layers. *
C      *---------------------------*
       NLAYERS = TGLAY

C      *------------------------------------*
C      * Set incoming flux at "layer" zero. *
C      *------------------------------------*
       WFLUX(0) = TGWFLUX(0)

C      *------------------------------------------------*
C      * Set up the following variables for each layer: *
C      * - bulk density                                 *
C      * - catchment                                    *
C      * - theta                                        *
C      * - thickness                                    *
C      * - water flux                                   *
C      * - root senscence                               *
C      * - soil moisture                                *
C      *------------------------------------------------*
       DO 100 LYR=1, NLAYERS
          BLKDENS(LYR)   = TGBULK(LYR)
          CATCHAREA(LYR) = TGCATCH(LYR)
          THETA(LYR)     = TGTHETA(LYR)
          THICKNESS(LYR) = TGTHICK(LYR)
          WFLUX(LYR)     = TGWFLUX(LYR)
          RTSENESC(LYR)  = TGRTSEN(LYR)
          SLTEMP(LYR)    = TGSLTEMP(LYR)
          SLMOIST(LYR)   = TGSLMOIST(LYR)
  100  CONTINUE

C      *-----------------------------*
C      * Set up the leaf senescence. *
C      *-----------------------------*
       LFSENESC = TGLFSEN

C      *---------------------------------------------*
C      * Set the number of species of interest, the  *
C      * name of each species of interest,  and the  *
C      * throughfall for each species of interest.   *
C      *---------------------------------------------*
       SPNUM = TGSPNUM
       DO 250 I=1,SPNUM
          SPNAME(I)    = TGSPNAME(I)
          NUTTFCONC(I) = TGTFCONC(I)
  250  CONTINUE

C      *--------------------------------------------------------------------*
C      * Indicate that detailed output for decomposition is to be written.  *
C      * This will eventually be an input parameter or this section will be *
C      * deleted.                                                           *
C      *--------------------------------------------------------------------*
       DCMPDETAIL = .TRUE.

C      *-----------------------------------------------------------*
C      * If this is the first time the soil model has been called, *
C      * open the output files.                                    *
C      *-----------------------------------------------------------*
       IF (FRSTIM.EQ.1) THEN
	         OPEN(UNIT = YASOUT, STATUS='NEW', FILE = OUTFILE)
       ELSE
C         *------------------------------------------------------------*
C         * Set up tree nutrients in senesced leaves and roots in YASE *
C         * order.                                                     *
C         *------------------------------------------------------------*
          DO 260 I=1,SPNUM
             LFNUT(SPINDEX(I)) = TGLFNUT(TGSPINX(I))
  260     CONTINUE
          DO 270 LYR=1, NLAYERS
             DO 280 J=1, TGSPNUM
                RTNUT(SPINDEX(J),LYR) = TGRTNUT(TGSPINX(J),LYR)
                NUTAKE(SPINDEX(J),LYR) = TGNUTS(TGSPINX(J),LYR)
  280        CONTINUE
  270     CONTINUE
	      ENDIF 

C      *----------------------------------------------------------------*
C      * Run the soil model. If there is an error condition exit model. *
C      *----------------------------------------------------------------*
       CALL SOILCHEM(GOAHEAD)
       IF (GOAHEAD.EQ.0) GOTO 9999

C      *---------------------------------------*
C      * Output data to file, pass dummy data. *
C      *---------------------------------------*
       LYR = 1
       CALL OUTRESULTS(YASOUT, LYR, LYR)

C      *-----------------------------------------------------------*
C      * Save the old theta to be used for water flux movement,    *
C      * fill the soil chemistry array as TreGro expects it, and   *
C      * fill the array of the total amount of nutrients in the    *
C      * soil as TreGro expects it.                                *
C      *-----------------------------------------------------------*
       DO 300 LYR=1, NLAYERS
          OLDTHETA(LYR) = THETA(LYR)
          DO 400 J=1, SPNUM
             SOCHEM(TGSPINX(J),LYR) = SPCONC(SPINDEX(J),LYR)
             SOTOT(TGSPINX(J),LYR)  = KT(SPINDEX(J),LYR)
  400     CONTINUE
  300  CONTINUE

C      *--------------*
C      * Escape hatch *
C      *--------------*
 9999  CONTINUE
	      IF (GOAHEAD.NE.1) THEN
	         WRITE(6,*) 'EXITING SOIL MODEL WITH ERROR CONDITION'
          PAUSE
       ENDIF

	      END
C      *****************************************************************
C      *                    END PROGRAM                                *
C      *****************************************************************

